<?php

namespace App\Http\Controllers\Auth;

use App\Enums\Status;
use App\Http\Controllers\Controller;
use App\Jobs\SendEmailVerificationJob;
use App\Enums\Email\EmailSmsTemplateName;
use App\Jobs\SendSmsVerificationJob;
use App\Models\User;
use App\Notifications\UserRegisteredNotification;
use App\Providers\RouteServiceProvider;
use App\Services\Payment\WalletService;
use App\Services\SettingService;
use App\Services\UserService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Validation\Rules;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;
use Lunaweb\RecaptchaV3\Facades\RecaptchaV3;

class RegisteredUserController extends Controller
{
    public function __construct(
        protected WalletService $walletService,
        protected UserService $userService,
    )
    {

    }
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        $setTitle = 'Register';
        $referral = $this->userService->getReferral();

        $setting = SettingService::getSetting();
        if (getArrayValue($setting?->system_configuration, 'registration_status.value') != Status::ACTIVE->value) {
            abort(404);
        }

        return view('auth.register', compact('referral', 'setTitle'));
    }
    
    

    /**
     * Handle an incoming registration request.
     *
     * @throws ValidationException
     */
     
     
     
     public function createfranchisee(): View
    {
        $setTitle = 'Register As Franchisee';
        $referral = $this->userService->getReferral();

        $setting = SettingService::getSetting();
        if (getArrayValue($setting?->system_configuration, 'registration_status.value') != Status::ACTIVE->value) {
            abort(404);
        }

        return view('auth.register-as-a-franchisee', compact('referral', 'setTitle'));
    }
     
     
     
public function store(Request $request): RedirectResponse
{
    $setting = SettingService::getSetting();

    // Check if registration is active
    if (getArrayValue($setting?->system_configuration, 'registration_status.value') != Status::ACTIVE->value) {
        abort(404);
    }

    // Validate reCAPTCHA if enabled
    if (getArrayValue($setting?->recaptcha_setting, 'registration') == Status::ACTIVE->value
        && RecaptchaV3::verify($request->input('g-recaptcha-response')) <= 0.3) {
        return back()->with('notify', [['error', 'Captcha verification failed']]);
    }

    // Validate the incoming request
    $request->validate([
        'name' => ['required', 'string', 'max:255'],
        'username' => ['required', 'string', 'min:5', 'max:255', 'unique:' . User::class],
        'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
        'phone' => ['required', 'numeric', 'digits:10', 'unique:' . User::class],
        'password' => ['required', 'confirmed', Rules\Password::defaults()],
        'referral' => ['required', 'string'], // Ensure referral is required
    ]);

    // Get referral user
    $referral = $this->userService->getReferral();

    // If no referral found, redirect back with error
    if (!$referral) {
        return redirect()->back()->withErrors(['referral' => 'A sponsor is required to register.'])->withInput();
    }

    // Get franchisee ID based on referral
    $get_franchisee_id = $this->userService->getFranchisee($referral?->id);

    // Create new user with KYC status set to inactive
    $user = User::create([
        'uuid' => Str::uuid(),
        'first_name' => $request->input('name'),
        'username' => $request->input('username'),
        'email' => $request->input('email'),
        'phone' => $request->input('phone'),
        'referral_by' => $referral?->id,
        'password' => Hash::make($request->input('password')),
        'kyc_status' => 2, // Set KYC status to inactive
    ]);

    // Update franchisee ID for the user
    User::where('id', $user->id)->update([
        'franchisee_id' => $get_franchisee_id,
    ]);

    // Perform email verification if enabled
    if (getArrayValue($setting->system_configuration, 'email_verification.value') == Status::ACTIVE->value) {
        SendEmailVerificationJob::dispatch($user);
    } else {
        $user->email_verified_at = now();
        $user->save();
    }

    // Perform SMS verification if enabled
    if (getArrayValue($setting->system_configuration, 'sms_verification.value') == Status::ACTIVE->value) {
        $code = random_int(100000, 999999);
        $user->verification_code = $code;
        $user->save();

        dispatch(new SendSmsVerificationJob($user, [
            'otp' => $code,
        ], $code, EmailSmsTemplateName::PHONE_NUMBER_VERIFICATION->value));
    } else {
        $user->phone_verified_at = now();
        $user->save();
    }

    // Save wallet details
    $this->walletService->save($this->walletService->prepParams((int) $user->id));

    // Notify user
    $user->notify(new UserRegisteredNotification());

    // Log in the user
    Auth::login($user);

    // Redirect to home page or desired route
    return redirect(RouteServiceProvider::HOME);
}


public function storefranchisee(Request $request): RedirectResponse
{
    $setting = SettingService::getSetting();

    if (getArrayValue($setting?->system_configuration, 'registration_status.value') != Status::ACTIVE->value) {
        abort(404);
    }

    if (getArrayValue($setting?->recaptcha_setting, 'registration') == Status::ACTIVE->value
        && RecaptchaV3::verify($request->input('g-recaptcha-response')) <= 0.3) {
        return back()->with('notify', [['error', 'Captcha verification failed']]);
    }

    $request->validate([
        'name' => ['required', 'string', 'max:255'],
        'username' => ['required', 'string', 'min:5', 'max:255', 'unique:' . User::class],
        'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
        'phone' => ['required', 'numeric', 'digits:10', 'unique:' . User::class],
        'password' => ['required', 'confirmed', Rules\Password::defaults()],
        'address' => ['required', 'string', 'max:255'],
        'country' => ['required', 'string', 'max:255'],
        'city' => ['required', 'string', 'max:255'],
        'postcode' => ['required', 'string', 'max:255'],
        'state' => ['required', 'string', 'max:255'],
    ]);

    $user = User::create([
        'uuid' => Str::uuid(),
        'first_name' => $request->input('name'),
        'username' => $request->input('username'),
        'email' => $request->input('email'),
        'phone' => $request->input('phone'),
        'password' => Hash::make($request->input('password')),
        'is_franchisee' => 1, // Updated as Franchisee
        'kyc_status' => 2, // Set KYC status to inactive
        'address' => json_encode([
            'address' => $request->input('address'),
            'country' => $request->input('country'),
            'city' => $request->input('city'),
            'postcode' => $request->input('postcode'),
            'state' => $request->input('state'),
        ]),
    ]);

    if (getArrayValue($setting->system_configuration, 'email_verification.value') == Status::ACTIVE->value) {
        SendEmailVerificationJob::dispatch($user);
    } else {
        $user->email_verified_at = now();
        $user->save();
    }
    if (getArrayValue($setting->system_configuration, 'sms_verification.value') == Status::ACTIVE->value) {
        $code = random_int(100000, 999999);
        $user->verification_code = $code;
        $user->save();
        
        dispatch(new SendSmsVerificationJob($user, [
            'otp' => $code,
        ], $code, EmailSmsTemplateName::PHONE_NUMBER_VERIFICATION->value));
    } else {
        $user->phone_verified_at = now();
        $user->save();
    }

    $this->walletService->save($this->walletService->prepParams((int) $user->id));
    $user->notify(new UserRegisteredNotification());

    Auth::login($user);
    return redirect(RouteServiceProvider::HOME);
}
}