<?php

namespace App\Http\Controllers\User;

use App\Concerns\CustomValidation;
use App\Enums\Payment\NotificationType;
use App\Enums\Payment\Withdraw\MethodStatus;
use App\Enums\Payment\Withdraw\Status;
use App\Enums\Transaction\Source;
use App\Enums\Transaction\Type;
use App\Enums\Transaction\WalletType;
use App\Http\Controllers\Controller;
use App\Http\Requests\WithdrawProcessRequest;
use App\Notifications\WithdrawNotification;
use App\Services\Payment\TransactionService;
use App\Services\Payment\WalletService;
use App\Services\Payment\WithdrawGatewayService;
use App\Services\Payment\WithdrawService;
use App\Services\UserService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\View\View;

class WithdrawController extends Controller
{
    use CustomValidation;

    public function __construct(
        protected WithdrawService $withdrawService,
        protected UserService $userService,
        protected TransactionService $transactionService,
        protected WalletService $walletService,
        protected WithdrawGatewayService $withdrawGatewayService
    ){

    }

    public function index(): View
    {
        $setTitle = "Cash out";
        $userId = Auth::id();
        $withdrawMethods = $this->withdrawGatewayService->fetchActiveWithdrawMethod();
        $withdrawLogs = $this->withdrawService->fetchWithdrawLogs(userId: $userId, with: ['user', 'withdrawMethod']);

        return view('user.withdraw.index', compact(
            'setTitle',
            'withdrawMethods',
            'withdrawLogs',
        ));
    }

    public function process(WithdrawProcessRequest $request)
    {
        $withdrawMethod = $this->withdrawGatewayService->findById($request->integer('id'));

        if(!$withdrawMethod){
            abort(404);
        }

        $amount = $request->input('amount');
        $validationError = $this->withdrawService->validateWithdrawalAmount($amount, $withdrawMethod, Auth::user()->wallet);

        if ($validationError != null) {
            return back()->withNotify([$validationError]);
        }

        $withdrawLog = $this->withdrawService->saveWithdrawLog(
            $this->withdrawService->withdrawLogPrepParams($withdrawMethod, $request)
        );

        return redirect()->route('user.withdraw.preview', $withdrawLog->uid);
    }


    /**
     * @param string $uid
     * @return View
     */
    public function preview(string $uid): View
    {
        $withdrawLog = $this->withdrawService->findByUidWithdrawLog($uid);

        if(!$withdrawLog){
            abort(404);
        }

        $setTitle = 'Withdraw preview';
        return view('user.withdraw.preview', compact(
            'setTitle',
            'withdrawLog',
            'uid',
        ));
    }


    /**
     * @param Request $request
     * @param string $uid
     * @return RedirectResponse
     * @throws ValidationException
     */
    public function makeSuccess(Request $request, string $uid): RedirectResponse
    {
        $withdrawLog = $this->withdrawService->findByUidWithdrawLog($uid);

        if(!$withdrawLog){
            abort(404);
        }

        $gateway = $withdrawLog->withdrawMethod;
        if (!$gateway ||  $gateway->status == MethodStatus::INACTIVE->value) {
            abort(404);
        }

        $this->validate($request, $this->parameterValidation((array)$gateway->parameter));

        $wallet =  Auth::user()->wallet;
        if ($withdrawLog->amount > $wallet->primary_balance) {
            return back()->with('notify', [['error', 'Your request amount is larger then your current priamry balance.']]);
        }

        DB::transaction(function () use ($withdrawLog, $request, $gateway, $wallet) {
            $withdrawLog->status = Status::PENDING->value;
            $withdrawLog->meta = $request->only(array_keys($gateway->parameter));
            $withdrawLog->save();

            $wallet->primary_balance -= $withdrawLog->amount;
            $wallet->save();

            $this->transactionService->save($this->transactionService->prepParams([
                'user_id' => Auth::id(),
                'amount' => $withdrawLog->amount,
                'wallet' => $this->walletService->findBalanceByWalletType(WalletType::PRIMARY->value, $wallet),
                'charge' => $withdrawLog->charge,
                'trx' => $withdrawLog->trx,
                'type' => Type::MINUS->value,
                'wallet_type' => WalletType::PRIMARY->value,
                'source' => Source::ALL->value,
                'details' => "Withdraw ".shortAmount($withdrawLog->final_amount)." ".$withdrawLog->currency." via ".$withdrawLog->withdrawMethod->name,
            ]));
        });

        $withdrawLog->notify(new WithdrawNotification(NotificationType::REQUESTED));
        return redirect(route('user.withdraw.index'))->with('notify', [['success', 'Withdraw request sent successfully.']]);
    }
}
