<?php

namespace App\Http\Controllers\User;

use App\Enums\Email\EmailSmsTemplateName;
use App\Enums\Matrix\PinStatus;
use App\Enums\Payment\NotificationType;
use App\Enums\Status;
use App\Enums\Transaction\Source;
use App\Enums\Transaction\Type;
use App\Enums\Transaction\WalletType;
use App\Http\Controllers\Controller;
use App\Jobs\EmailSmsProcessJob;
use App\Models\User;
use App\Notifications\DepositNotification;
use App\Notifications\WithdrawNotification;
use App\Services\Payment\DepositService;
use App\Services\Payment\TransactionService;
use App\Services\Payment\WalletService;
use App\Services\Payment\WithdrawService;
use App\Services\PinGenerateService;
use App\Services\SettingService;
use App\Services\UserService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class RechargeController extends Controller
{
    public function __construct(
        protected PinGenerateService $pinGenerateService,
        protected WalletService $walletService,
        protected TransactionService $transactionService,
        protected WithdrawService $withdrawService,
        protected DepositService $depositService,
        protected UserService $userService
    ) {

    }

    public function index(): View
    {
        $setTitle = "Recharge Your Wallet";
        $pins = $this->pinGenerateService->getPinsByPaginate(userId: Auth::id());

        return view('user.recharge', compact(
            'setTitle',
            'pins'
        ));
    }

    public function save(Request $request): RedirectResponse
    {
        $setting = SettingService::getSetting();

        if (getArrayValue($setting->system_configuration, 'e_pin.value') != \App\Enums\Status::ACTIVE->value){
            abort(404);
        }

        $request->validate([
            'pin_number' => ['required', Rule::exists('pin_generates', 'pin_number')],
        ]);

        try {
            $setting = SettingService::getSetting();
            $pinNumber = $this->pinGenerateService->findByPinNumber($request->input('pin_number'));
            
            if (!$pinNumber) {
                return back()->with('notify', [['error', 'Recharge E-pin not found']]);
            }

            if ($pinNumber->status == PinStatus::USED->value) {
                return back()->with('notify', [['error', 'Recharge E-pin already used.']]);
            }

            $user = $this->userService->findById(Auth::id());
            $wallet = $user->wallet;

            DB::transaction(function () use ($pinNumber, $wallet, $user) {

                $amount = $pinNumber->amount;

                $pinNumber->status = PinStatus::USED->value;
                $pinNumber->user_id = Auth::id();
                $pinNumber->save();

                $wallet->primary_balance += $amount;
                $wallet->save();

                $this->transactionService->save(
                    $this->transactionService->prepParams([
                        'user_id' => Auth::id(),
                        'amount' => $amount,
                        'type' => Type::PLUS,
                        'details' => 'Top up E-Pin with amount ' . getCurrencySymbol() . shortAmount($amount),
                        'wallet' => $this->walletService->findBalanceByWalletType(WalletType::PRIMARY->value, $wallet),
                        'source' => Source::ALL->value
                    ])
                );

                dispatch(new EmailSmsProcessJob($user, [
                    'amount' => $amount,
                    'pin_number' => $pinNumber->pin_number,
                    'currency' => getCurrencySymbol(),
                ], EmailSmsTemplateName::PIN_RECHARGE->value));


                $deposit = $this->depositService->save($this->depositService->depositPrepParams($amount));
                $deposit->notify(new DepositNotification(NotificationType::APPROVED));
            });

            return back()->with('notify', [['success', 'Balance has been added to your primary wallet']]);

        }catch (\Exception $exception){
            return back()->with('notify', [['error', 'An error occurred while recharge E-pin.']]);
        }
    }


    public function generate(Request $request): RedirectResponse
    {
        $request->validate([
            'amount' => ['required','numeric','gt:0'],
        ]);

        try {
            $setting = SettingService::getSetting();

            if(getArrayValue($setting->system_configuration, 'e_pin.value') == Status::INACTIVE->value){
                abort(404);
            }

            $user = Auth::user();
            $wallet = $user->wallet;
            $charge = (($request->input('amount') / 100) * getArrayValue($setting->commissions_charge, 'e_pin_charge'));
            $account = $this->walletService->findBalanceByWalletType(WalletType::PRIMARY->value, $wallet);
            $amount = $request->input('amount');

            if($amount + $charge > Arr::get($account, 'balance')){
                return back()->with('notify', [['warning', "Your primary account balance is insufficient for generate pin."]]);
            }

            DB::transaction(function () use ($request, $user, $wallet, $setting, $amount, $charge) {
                $params = $this->pinGenerateService->prepParams(1,$amount - $charge, "Created by a ".$user->fullname);

                foreach ($params as &$param) {
                    $param = Arr::add($param, 'set_user_id', $user->id);
                }

                $wallet->primary_balance -= $request->input('amount');
                $wallet->save();

                $this->pinGenerateService->save($params);
                $this->transactionService->save($this->transactionService->prepParams([
                    'user_id' => Auth::id(),
                    'amount' => $amount,
                    'type' => Type::MINUS,
                    'charge' => $charge,
                    'details' => 'Generate an E-pin worth '.getCurrencySymbol().shortAmount($amount - $charge),
                    'wallet' => $this->walletService->findBalanceByWalletType(WalletType::PRIMARY->value, $wallet),
                    'source' => Source::ALL->value
                ]));

                $withdrawLog = $this->withdrawService->saveWithdrawLog($this->withdrawService->withdrawParams($amount, $charge));
                $withdrawLog->notify(new WithdrawNotification(NotificationType::APPROVED));
            });

            return back()->with('notify', [['success', __('admin.pin.notify.create.success')]]);

        }catch (\Exception $exception){
            return back()->with('notify', [['error', 'An error occurred while generating E-pin.']]);
        }
    }



}
