<?php

namespace App\Http\Controllers\User;

use App\Enums\Trade\TradeType;
use App\Enums\Transaction\Type;
use App\Enums\Transaction\WalletType;
use App\Events\CryptoPriceUpdated;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Requests\TradeRequest;
use App\Services\Payment\WalletService;
use App\Services\SettingService;
use App\Services\Trade\ActivityLogService;
use App\Services\Trade\CryptoCurrencyService;
use App\Services\Trade\ParameterService;
use App\Services\UserService;
use App\Models\CompetitionParticipants;
use App\Models\Competition;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Carbon\Carbon;

class TradeController extends Controller
{
    public function __construct(
        protected WalletService $walletService,
        protected UserService $userService,
        protected ActivityLogService $activityLogService,
        protected CryptoCurrencyService $cryptoCurrencyService,
        protected ParameterService $parameterService,
    ){
    }

    public function index(): View
    {
        $setTitle = "Trades";
        $cryptoCurrency = $this->cryptoCurrencyService->getActiveCryptoCurrencyByPaginate();

        return view('user.trade.index', compact(
           'setTitle',
            'cryptoCurrency',
        ));
    }

    public function tradeLog(): View
    {
        $setTitle = "Trade logs";

        $userId = (int)Auth::id();
        [$days, $amount] = $this->activityLogService->dayReport($userId);
        $trade = $this->activityLogService->getTradeReport($userId);
        $tradeLogs = $this->activityLogService->getByUser($userId, TradeType::TRADE);

        return view('user.trade.trade_log', compact(
            'setTitle',
            'tradeLogs',
            'trade',
            'days',
            'amount',
        ));
    }

    public function practiceLog(): View
    {
        $setTitle = "Practice logs";
        $userId = Auth::id();
        $practiceLogs = $this->activityLogService->getByUser($userId, TradeType::PRACTICE);

        return view('user.trade.practice_log', compact(
            'setTitle',
            'practiceLogs',
        ));
    }

   /**
     * @param string $pair
     * @return View
     */
    public function trade(string $pair): View
    {
        $setting = SettingService::getSetting();
        if (getArrayValue($setting->system_configuration, 'binary_trade.value') != \App\Enums\Status::ACTIVE->value) {
            abort(404);
        }

        $setTitle = "Trade now";

        $userId = (int)Auth::id();
        $crypto = $this->cryptoCurrencyService->findByPair($pair);
        $parameters = $this->parameterService->activeParameter();
        $tradeLogs = $this->activityLogService->getByUser($userId, TradeType::TRADE, true);

        return view('user.trade.trading', compact(
            'setTitle',
            'crypto',
            'parameters',
            'tradeLogs',
        ));
    }

    /**
     * @param string $pair
     * @return View
     */
    public function practice(string $pair): View
    {
        $setting = SettingService::getSetting();

        if (getArrayValue($setting->system_configuration, 'practice_trade.value') != \App\Enums\Status::ACTIVE->value) {
            abort(404);
        }

        $setTitle = "Practice now";
        $userId = (int)Auth::id();
        $crypto = $this->cryptoCurrencyService->findByPair($pair);
        $parameters = $this->parameterService->activeParameter();
        $tradeLogs = $this->activityLogService->getByUser($userId, TradeType::PRACTICE, true);

        return view('user.trade.trading', compact(
            'setTitle',
            'crypto',
            'parameters',
            'tradeLogs',
        ));
    }

    /**
     * @throws Exception
     */
    public function store(TradeRequest $request, $id)
    {
        try {
            $parameter = $this->parameterService->findById($request->integer('parameter_id'));
            $crypto = $this->cryptoCurrencyService->findById((int)$id);

            if(!$parameter || !$crypto){
                abort(404);
            }
			
			if($request->integer('type') == TradeType::VIRTUAL_TRADE->value){
				$checkCompetitionTime = Competition::where('id', $request->input('competition_id'))->first();
				$currentDate = Carbon::now();
				$startDate = Carbon::parse($checkCompetitionTime->start_time);
				$endDate = Carbon::parse($checkCompetitionTime->end_time);
				if(!$currentDate->between($startDate, $endDate)){
					throw new Exception("Competition time is over.");
				}
				
				$chkCompetitionBalance = $this->walletService->chkCompetitionBalance($request->input('amount'),$request->input('competition_id'));
			}
			
            $walletType = $request->integer('type') == TradeType::TRADE->value ? WalletType::TRADE->value : ($request->integer('type') == TradeType::VIRTUAL_TRADE->value ? WalletType::VIRTUAL_TRADE->value : WalletType::PRACTICE->value);

            [$wallet, $account] = $this->walletService->checkWalletBalance($request->input('amount'), $walletType, true);
            $this->activityLogService->executeTrade($request, $wallet, $account, Type::MINUS, $parameter, $crypto);
			
			if($request->integer('type') == TradeType::VIRTUAL_TRADE->value){
				$user = (int)Auth::id();
				$balance = CompetitionParticipants::where('user_id', $user)->where('competition_id', $request->input('competition_id'))->first();
				CompetitionParticipants::where('user_id', $user)->where('competition_id', $request->input('competition_id'))->update([
					'competition_trade_amt'=>floatval($balance->competition_trade_amt) - floatval($request->input('amount'))
				]);
			}
			
            $notify[] = ['success', "Trade has been generated"];
            return back()->withNotify($notify);

        }catch (Exception $exception){

            $notify[] = ['warning', $exception->getMessage()];
            return back()->withNotify($notify);
        }
    }
    public function trade_details(Request $request)
    {
		$pair = $request->crypto_id;
        $userId = (int)Auth::id();
        $data['crypto'] = $this->cryptoCurrencyService->findByPair($pair);
        $data['parameters'] = $this->parameterService->activeParameter();
        $data['tradeLogs'] = $this->activityLogService->getByUser($userId, TradeType::PRACTICE, true);
		return response(['status' => 200, 'data' => $data]);
    }
}
